/* FILE: xdetect.c                              (D. Tottingham  04/28/91)

This is the driver for xdetect, the multi-channel event detection and data
collection program.  The program has been written and compiled medium model.

EXTERNAL FUNCTIONS CALLED:

dm_initialize ()                initialize the queues
dm_initialize_params ()         initialize parameters
dm_initialize_time ()           initialize the time
dm_reset ()                     reset this module and release all storage
dsp_spectral_detection ()       has a spectral been detected?
dsp_spectral_detection_done ()  are we still saving the spectral event?
dsp_get_dsp_status ()           get dsp_enabled
dsp_domain_check ()             look for a calibration signal
dsp_initialize ()               initialize the dsp module
dsp_initialize_params ()        initialize key parameters
dt_dump_configuration ()        dump the DT2821 configuration
dt_get_channel_size ()          get channel block size
dt_get_number_of_ext_buffers () get number of external buffers
dt_get_scan_count ()            get scan count
dt_initialize_ADC ()            initialize the DT2821
dt_initialize_mux ()            initialize multiplexer
dt_initialize_params()          initialize parameters
dt_set_mux_type ()              set the multiplexer type
dt_start_ADC ()                 start continuous data acquisition
dt_stop_ADC ()                  stop continuous data acquisition
e_initialize ()                 initialize event parameters
e_get_bell_status ()            get bell_enabled
e_ring_bell ()                  ring event alert bell if enabled
e_toggle_bell_status ()         toggle bell_enabled flag
er_abort ()                     display an error message then quit
f_check_pathname ()             check pathname for validity
f_initialize_index ()           initialize the index file
f_initialize_params ()          initialize key parameters
fr_initialize ()                initialize free run parameters
fr_toggle_freerun ()            toggle free run event
h_initialize_params ()          initialize a screen header
h_set_uptime ()                 update uptime.
h_toggle_status ()              toggle a status bit
l_display_location ()           write location to stream
l_get_location_status ()        get location_enabled
l_initialize ()                 initialize location module
l_locate_event ()               locate the event
l_reset_location ()             reset the location flag
l_toggle_location_status ()     toggle location_enabled flag
o_check_pathname ()             check log pathname for validity
o_initialize_params ()          initialize log pathname
o_open_logfile ()               open a log file for output
o_write_logfile ()              write string to log file stream
p_initialize ()                 initialize parser
p_parse_commands ()             parse commands in input stream
pk_initialize ()                initialize pick module
pk_pick_arrivals ()             pick arrivals using trigger information
pk_pick_magnitudes ()           pick magnitudes
pk_reset_picks ()               reset pick queues
r_check_time ()                 compare time with reboot time
r_get_reboot_status ()          get reboot_enabled
r_get_reboot_time ()            get reboot time
r_initialize_reboot ()          initialize reboot structure
r_toggle_reboot_status ()       toggle reboot_enabled flag
s_initialize ()                 initialize the screen module
s_initialize_screen ()          initialize the graphics screen
s_display_ids ()                display station id, channel, and trigger status
s_display_traces ()             display normalized traces on the screen
s_display_triggers ()           display triggers on the screen
s_reset_screen ()               reset the monochrome screen to text
s_select_display ()             select the display mode
s_set_channelrange ()           set the channel range in the block display
st_initialize_params ()         initialize station queue
st_initialize_stations ()       initialize station list
t_display_triggers ()           write event information to stream
t_get_trigger_status ()         get trigger_enabled
t_initialize_event ()           initialize the event structure
t_initialize_trigger ()         initialize the trigger structure
t_reset_trigger ()              reset trigger variables
t_toggle_trigger_status ()      toggle trigger_enabled flag
t_trigger_check ()              look for new triggers
u_build_timeline ()             convert abs. time to an ascii string
u_ispow2 ()                     determine whether i_num is a power of two
u_tzset_GMT ()                  set timezone to GMT
u_timestamp ()                  get timestamp

HISTORY:

(03/24/91) o Modified mdt28xx.c to support the new DT2827 64 X 4 multiplexer.
V2.01        Added the mux_configuration table and routines
             get_mux_descriptor(), get_mux_factor(), valid_multiplexer(), and
             valid_mux_id().

           o Created dt28xx_info table to support all DT2800 series A/D boards.
             Added routines get_dynamic_range(), compute_dc_offset(),
             dt_get_clip_value(), dt_get_data_type(), dt_get_data_units(),
             dt_get_dc_offset(), and dt_get_input_range().

           o Created gain_map and max_gain_map tables for gain validation.
             Added routines get_max_gain() and valid_gain().

           o Modified st_add_station() and st_set_gain() in mstation module.
             Added calls to dt_get_clip_value(), dt_get_dc_offset(),
             dt_get_input_range(), dt_get_data_type(), and dt_get_data_units().

           o Modified the SUDS_MUXDATA structure; spare word is now the dc
             offset.  Added dynamic range field to the Q_BUFFER structure.
             The dynamic range field is set by allocate_int_buffer() and
             dm_get_new_buffers().

           o Replaced ADC_MID_POINT with dc offset given in SUDS_MUXDATA
             structure.  Specifically, modified process_coda(),
             t_trigger_check(), and s_display_traces().

           o Modified s_display_traces() to properly scale data to an
             arbitrary dynamic range.

           o Added YES and NO constants and BYTE typedef to mconst.h.

           o Modified dt_set_channel() to return boolean flag to indicate
             valid channel.  Changed calll to dt_set_channel() in
             process_station() accordingly.  XDETECT now ignores invalid
             channels rather than aborting.

           o Added an uptime indicator to the screen header.  Added routine
             dm_get_uptime(), u_gmtime(), and h_set_uptime().  Added
             U_ELAPSED_TIME structure to mutils.h.  Added call to
             h_set_uptime() in process_buffer().

(03/03/91) o Modified mdt28xx.c to support the modified DT2821 128 X 8
V2.00        multiplexer.  In particular, added public routines
             dt_get_bankswitch_count(), dt_initialize_mux(), and
             dt_set_mux_type().  Added private routines send_mux_command(),
             clear_mux(), write_nbanks(), read_muxid(), and
             read_current_bank().  These routines implement the new MUX
             interface standard as defined by the DT2827 64 X 4 MUX design.
             This new interface standard supports auto MUX board detection
             and auto bank switch correction.  Much of the existing MUX
             software in this module was reworked to support this new interface
             standard.

           o Modified xdetect.c to support the new /Mxxx command line option.
             This option was added to support users that continue to use the
             old multiplexer without modification.
             xxx specifies the desired multiplexer type as follows:

                255 = old 128 X 16 multiplexer without modification
                  0 = old 128 X 16 multiplexer with modification
                  8 = new 64 X 4 multiplexer

             In addition, the routine initialize_command_line() was added to
             initialize the command line parameters, and process_input() was
             split into process_command_line() and process_input_file().

           o Modified mscrnhdr.c to display the number of bank switched buffers
             encountered within the current session.  To do this, the
             events field was expanded as follows:

                xxx events/ yyy bs

             where xxx is the number of events and yyy is the number of
             bank switched buffers.  As long as there are no bank switched
             buffers, the events field remains as before:

                xxx events

           o Added the bank_switched field to the Q_BUFFER structure in
             mqueue.h.  This field is set by dt_get_new_buffers() and
             dm_get_new_buffers().

           o Modified mtrigger.c (t_trigger_check() in particular) to
             ignore bank switched buffers.  To do this, the t_trigger_check()
             routine looks at the bank_switched flag in the Q_BUFFER.

           o Added DT_IO_NOTINPROG error to merror.h and merror.c.

(01/26/91) o Fixed bug in pwrlolvl.asm which was causing intermittent
V1.99        'Invalid PowrSTOR address' errors.  The function putSTOR would call
             chkAddr to determine whether buffer overlapped a segment
             boundary.  To determine this overlap, the chkAddr function added
             the length of the buffer to the base address and then looked at
             the overflow flag.  Using the overflow flag meant that any buffer
             overlapping the middle of the segment was in error.  The carry bit
             is set up for just this purpose.  If the result exceeds 0xffff, the
             carry flag is set.  So I changed a JNO instruction to a JNC
             instruction.

(12/26/90) o Created the mfreerun module to handle/manage free run events.
V1.98        The FreerunBlockTime parameter was added to the input file to
             allow the user to block free run events into smaller, manageable
             events.

           o Created the mevent module to act as a sophisticated event manager.
             As of now, the event manager just manages the event alert bell.

           o Replaced the x_set_bell_status(), x_set_reboot_status(),
             x_set_locate_status(), and x_set_trigger_status() routines in
             xdetect.c with appropriate routines in the mevent, mreboot,
             mlocate, and mtrigger modules respectively.  Eliminated the
             bell_enabled, reboot_enabled, location_enabled, and trigger_enabled
             flags from xdetect.c.

           o Miscellaneous cleanup.

(12/02/90) o Fixed bug in dm_initialize().  Due to the fact that PowerSTOR
V1.97        allocates memory in 16K byte chunks, changed the way that
             dm_initialize() calculates the number of 16K byte chunks in
             pre-trigger queue.

(11/02/90) o Made changes to mdemux.c to support larger pre-trigger queues. In
V1.96        particular, added code to move the pre-trigger queue to extended
             memory using the PowerSTOR subroutine library.

           o Fixed a bug in the rms residual calculation.

           o Changed the way the mscreen module allocates memory on the far
             heap.  We now assume that 32 channels (MAX_DISPLAY) will be
             displayed.  What would happen is that we would initially allocate
             storage for 32 channels in LIST mode, deallocate the storage when
             the user switched to BLOCK mode, and then try to reallocate the
             storage when the user switched back to LIST mode.  Unfortunately,
             while the user was in BLOCK mode, the additional storage used in
             LIST mode would be taken up by other processing thereby fragmenting
             the far heap.  XDETECT would then crash because it could not
             reallocate the storage.  This solution will make do until we come
             up with a better way to manage the far heap.

(09/29/90) o Added rms residual calculation to hs_locate_event().  Added
V1.95        rms residual to l_display_location().

           o Added functionality to dt_set_channel() and dt_set_ChannelGain()
             to make sure that the gain is 1, 2, 4, or 8.

           o Added functionality to u_ispow2() and u_log2() to make sure that
             they only operate on positive, non-zero numbers.

(09/08/90) o Fixed bug in u_get_diskfree().  The default drive was not being
V1.95        recognized.

           o Fixed bug in s_display_traces().  Negative data values were being
             scaled (amplified/attenuated) incorrectly resulting in spurious
             DC offsets.

           o Error messages are now routed to the log file.  Log file path
             now defaults to the current working directory.  Alternate log file
             path names (defined in the input file) are now ignored, and the
             appropriate warning message is displayed.  This is due to the fact
             that warning and/or error messages can be generated before reaching
             the log file path name definition in the input file resulting in
             two separate log files, one in the current working directory and
             the other in the log file path defined in the input file.             and will be sent to a log file residing within the current working

(03/30/90) o Added the STAverageWindow and LTAverageWindow parameters to the
V1.94        input file.  These parameters set the number of samples within
             the short-term average and long-term average (within the trigger
             algorithm) respectively.

           o Eliminated the NumberOfExtBuffers parameter from the input file.
             The mdt28xx module adds buffers until it gets an error from
             ATLAB.

           o ChannelBlocksize now defaults to 32768 / # of channels in station
             list.

           o Added timestamp to entries in log file.

           o Added error numbers to error messages.

(12/29/89) o Renamed msuds.c msudsini.c.  msudsini.h includes suds.h which
V1.93        defines all suds structures.  suds.h is centrally located in
             \msc51\include.

           o Finished integration of mdsp module into xdetect.  Added
             CalibrationRecording, BandRecording, and BandLimits keywords
             to mlexer.h and mlexer.c.  Added the following functionality to
             mparse.c:

              CalibrationRecording= [ON | OFF];
              BandRecording= [ON | OFF][; | , BandLimits= { (a, b),...,(a, b) };

             Removed DSPEnabled keyword.  If calibration recording is on and
             band recording is on, mparse sets dsp_enabled to TRUE in xdetect.c
             via x_set_dsp_status().

             Added H_RECORD_CALIBRATION bit field to mscrnhdr.h.  When a
             calibration is detected and is being saved, mscrnhdr displays
             "Recording Calibration" on the status line on top of the screen.

             Added dsp_set_band() and dsp_initialize_params() to mdsp module.
             dsp_set_band() accepts a band specification (high and low freq.
             cutoffs) and puts is into a new BAND structure in the array of
             BAND structures.  dsp_initialize_params() initializes the BAND
             structure array and operation parameters.

             Modified f_write_buffers() so that the event index is only
             incremented for events not calibrations.

             Cleaned up mdsp module.  Generated function documentation,
             alphabetized public mdsp functions, and rewrote small pieces of
             inadequate code.  Added error checking interfacing with the
             merror module.  Wrote init_tms320() to look for a TMS32xxx board.

           o Changed keywords and functionality of the following:

              BellEnabled= [TRUE | FALSE]        to  EventAlertBell= [ON | OFF]
              MonitoringEnabled= [TRUE | FALSE]  to  Autotriggering= [ON | OFF]
              LocationEnabled= [TRUE | FALSE]    to  AutoLocation= [ON | OFF]

              RebootEnabled= [TRUE | FALSE] and
              RebootTime= [xx:xx:xx]             to
                   Autoreboot= [ON | OFF] [; | , Time= [xx:xx:xx] ]

           o Changed status line text in mscrnhdr.c as follows:

              "Monitoring Enabled"   to   "Autotriggering"
              "Monitoring Disabled"  to   "Asleep"

           o Fixed coda duration based magnitude problem.  Instead of looking
             for the first sample in which signal/noise < 2.0, look for first
             buffer in which maximum signal/noise < 2.0.  Modified process_coda()
             in mpick.  Eliminated signal calculation in triggered_on_channel()
             in mtrigger.

           o Added XDETECT revision date and number to log file.

           o Added memory_dump flag to ER_MESSAGE handler in merror.h.  Added
             appropriate memory dump flags to each error message in merror.c.
             For every error, er_abort checks the memory dump flag in the
             error's ER_MESSAGE structure.  If it is set, dumps of both the
             near and far heap are directed to dumpheap.@@@ for future
             debugging.

(11/24/89) o Reduced minimum noise (NOISE_INIT) from 30 to 1 digital counts
V1.92        in mtrigger.h to get a more accurate magnitude determination.

           o Removed _dos_getdiskfree() and _dos_getdrive() from mdosfunc
             library and changed mdosfunc.h accordingly.  Replaced
             DISKFREE_T with struct diskfree_t in u_get_diskspace() in
             mutils.c.

           o Replaced mevt and mevtdrv modules with mdsp module and improved
             the DSP interface.  Replaced x_set_fft_status() with
             x_set_dsp_status() and fft_enabled with dsp_enabled.  Added dsp
             control code to process_buffer().  Removed fft code from
             post_trigger_analysis().  Added MaxCalibrationTime to input file.

           o Added f_get_pathname() to mfile module.

(09/15/89) o Added picker module (mpick.c).  This picker has two primary
V1.91        functions: pick arrivals and pick coda durations.  The pick
             arrival function simply takes a linked list of SUDS_TRIGGER
             structures from the trigger module and converts them into a linked
             list of SUDS_FEATURE (obs_phase = 50) structures.  The pick coda
             duration function determines the coda duration for each channel
             that has an arrival pick based on the signal-to-noise ratio.

           o Replaced the hypocenter location function with a halfspace
             function.  The mlocate module is still the driver for the
             location algorithm, but it now supports the SUDS_ORIGIN structure
             and maintains a linked list of the last ten locations.  This
             latter ability will be essential for plotting epicenter maps in
             real-time.

           o Added the mlatlon module to handle all converts to/from latlon
             from/to xy.

           o Added a vertical scroll feature to the trace display screen which
             allows the user to view selected 16 channel chunks in detail.
             You can enter the scroll mode by pressing either CNTRL-PGUP or
             CNTRL-PGDN.  CNTRL-PGUP and CNTRL-PGDN display the previous 16
             channels and the next 16 channels respectively.  Hitting
             CNTRL-HOME gets you back to the channels selected in the input
             file.  While you are in scroll mode, a scroll bar appears on the
             left side of the screen indicating which channels you are viewing
             with respect to the entire channel list.

           o Added a help screen.  Press F1 to display the help screen and any
             key to return to the trace display screen.

(08/24/89) o Changed file format to SUDS (Seismic Unified Data System) which
V1.90        was developed by Peter Wards.  Developed to be modular in
             structure, SUDS is flexible enough to handle as many or as few
             channels as the user needs.  In addition, modules can be easily
             added and modified without damaging the integrity of the file
             structure.  SUDS is now really the backbone of xdetect.

           o Added write_to_buffer() routine to the mfile module to buffer
             SUDS structures to disk.  Compared to the unbuffered approach,
             this method gives us a 100% speed improvement.

           o Modified s_display_traces() to scroll when buffers are less than
             512 samples in size.  We can use this function later on to
             let the user specify a decimation factor in real-time!!

           o Changed the input file slightly.
             a. Changed the station definition as such:
                 Ch= , StName= , Component= [, Trigger= ][, Display= ][, Gain= ];
                The brackets indicate optional parameters.  The order of the first
                two parameters is critical, and each parameter MUST be separated
                by a comma!!!  Default trigger is ON, default display is OFF, and
                default gain is either ChannelGain or 4 (CHANNEL_GAIN) if
                ChannelGain is not defined.

             b. Added NetworkName.  Specifies the network operating the stations
                defined in the station list.  XDETECT can only support one
                network!!!

             c. Added NetworkNodeId.  Allows you to have multiple computers
                monitor a single network.  This is simply an id that is pasted
                to every output file to uniquely identify it from other files
                generated by other computers in the network.  The first
                character of the NetworkNodeId is used as the network code and
                is appended to the filetype (i.e. .WVx where x is the network
                code).

             d. Added ChannelGain.  The gain parameter in the station definition
                is only valid on channels 0-15, and these are only valid if
                less than 17 channels are defined.  Otherwise, all gains are
                set to ChannelGain.  If ChannelGain is not given, then the
                default channel gain is used (CHANNEL_GAIN = 4).

             e. Changed ChannelBufferSize to ChannelBlocksize.

           o Added support for a station file, a file of SUDS_STATIONCOMP
             structures.  Each station has a large amount of information
             that should be written to every event file.  Due to the
             volume and static nature of this data, forcing the user to
             enter all this information for each station in the input file
             seems meaningless.  By supporting a separate, binary file of
             this station data, xdetect only expects a network name, station
             name, and component information for each station.  xdetect then
             finds the corresponding SUDS_STATIONCOMP structure in the
             station file for each station defined in the input file.
             If the station is not found, xdetect creates a dummy
             SUDS_STATIONCOMP and fills it with default/dummy values.
             Created STPARSE so that users can define SUDS_STATIONCOMP
             structures in ascii and then parse them into binary suds
             structures (see stparse.c).

           o Changed PreEventTime, MinEventTime, MaxEventTime, and
             TriggerTimeLimit to floats in order to support < 1 second
             intervals!!!

           o Rewrote merror to make it easier to add/remove error messages
             from the error message table.  Improved error handling for ATLAB:
             Check to see if card is in machine and if atldrv is installed.
             Make sure that low frequency and high frequency bounds are not
             crossed.

           o Made HYPO20 into a library.  Support MS Fortran V5.0.

           o Timer module now supports four independent timers.

(08/01/89) o Revamped the MTRIGGER module.  Set up two queues, one for
V1.81        channel information and one for trigger information.  Made changes
             to MQUEUE to handle a queue of anything (see Q_TYPE union).
             Eliminated the clumsy array handling and replaced it with linked
             list manipulation.  Channels can now have multiple triggers.
             The oldest trigger remains active until it expires at which time
             the next oldest trigger becomes active.

           o Added a display queue to hold channels which are to be displayed.
             Can handle up to 32 channels/screen.  Added a display parameter
             to the station definition in the input file to allow the user to
             select which 32 channels he wants to display.  Tested software
             and demonstrated 64 channel capability.

           o Changed screen layout.  Display with plotmod so we don't have to
             clear the screen for every buffer.  Added real-time
             amplify/attenuate feature as implemented in XPLAY.

(07/28/89) o Fixed the 32Kbyte limitation for internal buffers by changing the
V1.80        ATLAB device driver.  Basically, internal buffers do not have to
             meet DMA requirements because they are never accessed by DMA.
             However, because ATLAB allows you to have a buffer transfer list
             either in base memory or extended memory, Data Translation designed
             AL_DECLARE_BUFFER assuming that you will always have a local BTL.
             I commented out two lines from aldthr.icl so this test never occurs.

           o Renamed the MQUEUE module MDEMUX and moved all the queue manipulation
             routines to MQUEUE.  Set up a queue of internal mux buffers (my own
             BTL) each capable of holding 32Kbytes.  By doing this, I expanded
             the virtual buffer size to 64Kbytes (ATLAB limitation).

           o Changed the screen layout slightly in order to support all three
             monitors (Hercules, EGA, and VGA).  Now we support all three
             monitors.  Created three .LNK files, one for each monitor.

(07/20/89) o Migrated code to Microsoft C V5.1.  The .EXE file is now down to
V1.71        133Kbytes (from 233Kb), and execution speed has nearly doubled.

           o Added code to XDETECT.C to time acquisition cycle and to
             compute an approximate maximum digitization frequency.  XDETECT
             displays this approximate maximum digitization frequency on the
             screen just before exiting the program.

(07/14/89) o Source code has been completely reorganized into object oriented
V1.70        modules.  Each module has marked its public functions with a unique
             1 or 2 letter code.  For the most part, the code is simply the
             first letter of the module name (i.e. t_ prefixes all mtrigger
             routines).  See filemap.xxx for the code-module mapping.

           o Fixed the Control-Q problem.  In Version 1.6 and earlier,
             a user could quit the program at any time simply by pressing
             CNTRL-Q.  This caused problems when the program was recording
             an event (either triggered or in free run).  Now, a CNTRL-Q is
             interpreted as a flag, and the flag is not checked until after
             xdetect has finished recording the event.

           o Added Control-B.  This enables/disables the warning bell.
             Added BellEnabled token to control file.

           o Eliminated the ascii .IXM files and replaced them with a single
             binary INDEX.@@@ file.  To minimize problems in the future, this
             file is OFF LIMITS to post-processing programs.  The format of the
             file is currently:
                        day             (2 bytes)
                        month           (2 bytes)
                        year            (2 bytes)
                        index high word (2 bytes)
                        index low word  (2 bytes)

           o The screen header has been changed as follows:
                a. The event counter is reset at the beginning of each
                   session.
                b. Changed "Event Not Detected" to "Monitoring Enabled"
                c. Changed "Event DETECTED"     to "Recording Event"
                d. Changed "Trigger Disabled"   to "Monitoring Disabled"

           o The channel numbers are now displayed in decimal instead of
             hexadecimal.

           o Replaced the clumsy input file handler with a sophisticated
             parser/lexer.  This allows full syntax checking.  In order to
             use this parser, the input file syntax had to be changed slightly
             which makes it incompatible with previous versions.
             The changes are slight and are as follows:
                a. Any non-alphanumeric character string must be placed in
                   double quotes (i.e. StationId = "CH_1"; the underscore
                   makes the string non-alphanumeric).  This rule affects
                   StationID, PathName, and LogPathName.
                b. The station definition must have the following syntax:
                      StationId=  , Ch=  [, Gain=  ][, Trigger= ];
                   The brackets indicate optional parameters.
                   The order of the first two parameters is critical, and
                   each parameter MUST be separated by a comma!!!
                c. Any non-alphabetic character constant must be placed in
                   single quotes (i.e. NetworkCode= '0'; the 0 is
                   non-alphabetic).  This rule only affects NetworkCode.

           o Fixed the greenwich mean time problem.  In Version 1.6 and earlier,
             xdetect assumed that users were operating the system under local
             time.  Unfortunately, Microsoft coded their run-time library time
             routines in such a way that daylight savings time is automatically
             set/reset at undocumented, predefined dates in the Spring/Fall.
             xdetect now assumes that users are operating the system under GMT
             without daylight savings time.  Simply trick the Microsoft RTL into
             thinking that the system is at GMT by setting the global variables
             timezone = 0, daylight = 0, and tzname[0] = "GMT".

           o Changed TriggerEnabled to MonitorEnabled in control file.

           o Changed Control-T to Control-M, added Control-H to toggle HYPO,
             added Control-R to toggle REBOOT, and eliminated Control-P.

*/


/*************************************************************************
                             INCLUDE FILES

*************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <atldefs.h>

#include "mconst.h"
#include "mdemux.h"
#include "mdsp.h"
#include "mdt28xx.h"
#include "merror.h"
#include "mevent.h"
#include "mfile.h"
#include "mfreerun.h"
#include "mlocate.h"
#include "mlog.h"
#include "mparse.h"
#include "mpick.h"
#include "mreboot.h"
#include "mscreen.h"
#include "mscrnhdr.h"
#include "mstation.h"
#include "mtrigger.h"
#include "mutils.h"
#include "xdetect.h"
#include "timer.h"


/*************************************************************************
                              GLOBAL DATA

**************************************************************************/
FLAG Debug_enabled;                             /* debug on/off */
PRIVATE FLAG quit_enabled = FALSE;              /* quit flag */
PRIVATE unsigned int display_type;

PRIVATE char out_str[85];
double min_time, max_time, avg_E;


/*************************************************************************
                         FUNCTION DECLARATIONS

**************************************************************************/
PRIVATE void quit ();



/*=======================================================================*
 *                      initialize_command_line                          *
 *=======================================================================*/
/* Initialize command line parameters.                                   */

PRIVATE
void initialize_command_line ()
{
   Debug_enabled = DEBUG_ENABLED;
   dt_initialize_mux ();
}

/*=======================================================================*
 *                        initialize_defaults                            *
 *=======================================================================*/
/* Initialize default parameters.                                        */

PRIVATE
void initialize_defaults ()
{
   u_tzset_GMT ();
   st_initialize_params();
   dm_initialize_params();
   dsp_initialize_params();
   dt_initialize_params();
   e_initialize ();
   f_initialize_params();
   fr_initialize();
   h_initialize_params();
   l_initialize();
   o_initialize_params();
   p_initialize();
   pk_initialize();
   r_initialize_reboot();
   s_initialize();
   t_initialize_event();
   t_initialize_trigger();
}


/*=======================================================================*
 *                       process_command_line                            *
 *=======================================================================*/
/* Process the command line parameters.                                  */

PRIVATE
void process_command_line (filename, argc, argv)
char filename[];
int argc;
char * argv[];
{
   FLAG found_filename;

   /* Search for command line options and the filename */
   found_filename = FALSE;
   while (--argc) {
      switch (argv[argc][0]) {
         case '/':
            switch (toupper(argv[argc][1])) {
               case 'D':
                  Debug_enabled = TRUE;
                  break;

               case 'M':
                  dt_set_mux_type (atoi(&(argv[argc][2])));
                  break;

               /* room for other slash options */
            }
            break;
         default:
            strcpy (filename, argv[argc]);
            found_filename = TRUE;
      }
   }

   if (! found_filename) {
      printf ("Enter control file [<CR> = %s]  >> ", INPUT_FILENAME);
      gets (filename);
      printf ("\n");
      if (filename[0] == '\0')
         strcpy (filename, INPUT_FILENAME);
   }

}

/*=======================================================================*
 *                        process_input_file                             *
 *=======================================================================*/
/* Process the input file.                                               */

PRIVATE
void process_input_file (filename)
char filename[];
{
   FILE * inp_stream;

   o_write_logfile ("<<<<<<<<<<<<<<<<<<<<<   N E W   S E S S I O N   >>>>>>>>>>>>>>>>>>>>>\n");
   sprintf (out_str, "XDETECT Version %5.2f, Last Revision %s\n", VERSIONNUM, REVISION_DATE);
   o_write_logfile (out_str);
   sprintf (out_str, "INPUT FILE:  %s\n\n", filename);
   o_write_logfile (out_str);

   /* If the input file exists, process the contents */
   if (inp_stream = fopen (filename, "r")) {
      p_parse_commands (inp_stream);
      fclose (inp_stream);
   } else er_abort (X_INVALID_INPUTFILE);
}

/*=======================================================================*
 *                           display_setup                               *
 *=======================================================================*/
/* Display setup information.                                            */

PRIVATE
void display_setup ()
{
   /* Print out some key flags and values */
   printf ("\nBell      DSP       Location  Trigger   Reboot    Reboot time   Clk source\n");
   o_write_logfile ("Bell      DSP       Location  Trigger   Reboot    Reboot time   Clk source\n");
   sprintf (out_str, "%-10s%-10s%-10s%-10s%-10s%-8s      %-14s\n\n",
           ((e_get_bell_status())       ? "ENABLED"  : "DISABLED"),
           ((dsp_get_dsp_status())      ? "ENABLED"  : "DISABLED"),
           ((l_get_location_status())   ? "ENABLED"  : "DISABLED"),
           ((t_get_trigger_status())    ? "ENABLED"  : "DISABLED"),
           ((r_get_reboot_status())     ? "ENABLED"  : "DISABLED"),
           r_get_reboot_time(),
           ((dt_get_clock_source() == 'i') ? "INTERNAL" : "EXTERNAL"));

   printf (out_str);
   o_write_logfile (out_str);
   sprintf (out_str, "Number of external buffers = %d\n\n", dt_get_number_of_ext_buffers());
   printf (out_str);
   o_write_logfile (out_str);
}

/*=======================================================================*
 *                          check_pathnames                              *
 *=======================================================================*/
/* Check pathnames.                                                      */

PRIVATE
void check_pathnames ()
{
   o_check_pathname();
   f_check_pathname();
}

/*=======================================================================*
 *                         initialize_system                             *
 *=======================================================================*/
/* Initialize system parameters.                                         */

PRIVATE
void initialize_system ()
{
   /* Initialize the station list */
   st_initialize_stations ();

   /* Initialize the DSP */
   dsp_initialize();

   /* Initialize the ADC */
   dt_initialize_ADC ();

   /* Allocate internal and external multiplexed buffers */
   dt_initialize_buffers();

   /* Initialize the demux queue.  Setup the PowerSTOR */
   dm_initialize ();

   /* Start data acquisition */
   dt_start_ADC ();

   /* Display the current setup */
   display_setup ();

   /* Initialize the screen and screen header */
   s_initialize_screen ();
   dm_initialize_time ();

   /* Initialize the index file */
   f_initialize_index ();

   /* Initialize the ids */
   display_type = HOME;
   s_select_display (HOME);
   s_display_ids ();
}

/*=======================================================================*
 *                       post_trigger_analysis                           *
 *=======================================================================*/

PRIVATE
void post_trigger_analysis ()
{
   FILE * log_stream;

   /* Display triggers on screen. */
   if (Debug_enabled)
      t_display_triggers (stdout);
   if (LOG_FILE) {
      log_stream = o_open_logfile ();
      t_display_triggers (log_stream);
      fclose (log_stream);
   }
}

/*=======================================================================*
 *                           process_buffer                              *
 *=======================================================================*/
/* Main processing loop.                                                 */

PRIVATE
void process_buffer ()
{
   FILE * log_stream;
   double dt;

   /* Do we have a buffer to process */
   if (! dt_buffer_full())
      return;

   /* Main processing loop */
   set_starttime(TIMER_0);
   dm_get_new_buffers();
   s_display_traces (0, 0);
   h_set_uptime ();

   if (dsp_spectral_detection()) {
      if (dsp_spectral_detection_done()) {
        f_write_buffers (END_FILE, F_CALIBRATION);
        h_toggle_status (H_RECORD_CALIBRATION);
      }
      else f_write_buffers (CONTINUE_FILE, F_CALIBRATION);

   } else if (t_event_detected()) {
      if (t_event_done ()) {
         /* Event is over.  Locate the event */
         if (l_locate_event()) {
            if (LOG_FILE) {
               log_stream = o_open_logfile ();
               l_display_location (log_stream);
               fclose (log_stream);
            }
            if (Debug_enabled) l_display_location (stdout);
         }
         f_write_buffers (END_FILE, F_EARTHQUAKE);
         l_reset_location ();
         pk_reset_picks ();
         t_reset_trigger ();
         h_toggle_status (H_RECORD_EVENT);
         if (Debug_enabled)
            printf ("Data Acquisition continued...\n");

      } else {
         f_write_buffers (CONTINUE_FILE, F_EARTHQUAKE);
         pk_pick_magnitudes ();
      }

   } else if (fr_continue_freerun()) {

   } else if (quit_enabled) {
        quit();

   } else if (r_check_time()) {

   } else if (dsp_domain_check()) {
      if (Debug_enabled)
         printf ("Calibration found and being saved...\n");
      h_toggle_status (H_RECORD_CALIBRATION);
      f_write_buffers (START_FILE, F_CALIBRATION);

   } else if (t_trigger_check ()) {
      e_ring_bell ();

      /* Setup to collect eq data */
      h_toggle_status (H_RECORD_EVENT);

      /* Pick the p-arrivals and magnitudes */
      pk_pick_arrivals ();
      pk_pick_magnitudes ();

      /* Do some post-trigger processing at this point */
      post_trigger_analysis ();

      /* Write pre-trigger data to disk */
      f_write_buffers (START_FILE, F_EARTHQUAKE);
   }

   set_stoptime(TIMER_0);
   dt = diff_time(TIMER_0);
   if (dt < min_time) min_time = dt;
   if (dt > max_time) max_time = dt;
   if (t_event_detected() /* || free_run_flag */)
      avg_E += (dt - avg_E) / dt_get_number_of_ext_buffers();
}

/*=======================================================================*
 *                               quit                                    *
 *=======================================================================*/
/* Reset screen to text.  Quit xdetect.                                  */

PRIVATE
void quit ()
{
   double avg_time, max_sps;

   dt_stop_ADC ();
   s_reset_screen ();
   dm_reset();

   printf ("\n\nMinimum time               = %lf sec.\n", min_time);
   printf ("Maximum time               = %lf sec.\n", max_time);
   if (avg_E) {
      max_sps = dt_get_channel_size() / avg_E;
      printf ("Average event time         = %lf sec.\n", avg_E);
      printf ("Maximum digitization rate  = %lf sps.\n", max_sps);
   }

   o_write_logfile ("..... End of session .....\n");

   exit ( 1);
}

/*=======================================================================*
 *                                 m a i n                               *
 *=======================================================================*/
main (argc, argv)
int argc;
char * argv[];
{
   FILE * log_stream;
   char * timeline;
   char filename[MAX_FILENAME_LENGTH];
   int c;

   s_reset_screen ();

   /* Opening banner */
   printf ("\n\n%s (%s)    Tottingham\n", REVISION_NAME, REVISION_DATE);
   printf ("\nMulti-Channel Event Detection and Collection Program\n\n");

   initialize_command_line ();

   process_command_line (filename, argc, argv);

   initialize_defaults ();

   process_input_file (filename);

   check_pathnames ();

   initialize_system ();

   /* Display the current unit configuration */
   if (Debug_enabled)
      dt_dump_configuration (stdout);
   if (LOG_FILE) {
      log_stream = o_open_logfile();
      dt_dump_configuration (log_stream);
      fclose (log_stream);
   }

   min_time = 10000;  /* just a big number */
   max_time = avg_E = 0;

   while (1) {
      if (c = keystat ()) {
         switch (c) {
            case CNTRL_B:
               e_toggle_bell_status ();

               if (Debug_enabled)
                  printf ("Bell %s \n",
                          ((e_get_bell_status()) ? "ENABLED" : "DISABLED"));
               break;
            case CNTRL_F:
               fr_toggle_freerun();
               break;
            case CNTRL_L:
               l_toggle_location_status ();
               if (Debug_enabled)
                  printf ("Location %s \n",
                          ((l_get_location_status()) ? "ENABLED" : "DISABLED"));
               break;
            case CNTRL_HOME:
               if (display_type != HOME) {
                  s_select_display (HOME);
                  s_display_ids ();
                  s_display_traces (0, 0);
                  s_display_triggers ();
                  display_type = HOME;
               }
               break;
            case CNTRL_PGDOWN:
               if (display_type == BLOCK &&
                   !s_set_channelrange (MAX_BLOCK, MAX_BLOCK))
                  continue;
               else if (display_type != BLOCK) {
                  s_select_display (BLOCK);
                  display_type = BLOCK;
                  s_set_channelrange (0, MAX_BLOCK);
               }
               s_display_ids ();
               s_display_traces (0, 0);
               s_display_triggers ();
               break;
            case CNTRL_PGUP:
               if (display_type == BLOCK &&
                   !s_set_channelrange (-MAX_BLOCK, MAX_BLOCK))
                  continue;
               else if (display_type != BLOCK) {
                  s_select_display (BLOCK);
                  display_type = BLOCK;
                  s_set_channelrange (0, MAX_BLOCK);
               }
               s_display_ids ();
               s_display_traces (0, 0);
               s_display_triggers ();
               break;
            case CNTRL_Q:
               quit_enabled = TRUE;
               break;
            case CNTRL_R:
               r_toggle_reboot_status ();

               if (Debug_enabled)
                  printf ("Reboot %s \n",
                          ((r_get_reboot_status()) ? "ENABLED" : "DISABLED"));
               break;
            case CNTRL_T:
               t_toggle_trigger_status ();

               h_toggle_status (H_AUTOTRIGGER_ENABLED);
               if (Debug_enabled)
                  printf ("Autotriggering %s \n",
                          ((t_get_trigger_status()) ? "ENABLED" : "DISABLED"));
               break;
            case DOWN_ARROW:
               s_display_traces (0, -1);
               break;
            case F1:
               s_view_help ();
               while (! keystat())
                  process_buffer ();
               s_clear ();
               s_select_screen (SCREEN_1);
               h_update ();
               s_display_ids ();
               s_display_traces (0, 0);
               s_display_triggers ();
               if (display_type == BLOCK)
                  s_display_bar ();
               break;
            case UP_ARROW:
               s_display_traces (0, 1);
               break;
            default:
               break;
         }
      }
      process_buffer ();
   }
}
